%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% Function name: bin_data
%
% Scope:    integrate raw data at a given time resolution
%
% Input:    data (structure),
%           data_ch4 (structure),
%           list of gas names (cell),
%           list of gas type "flags" (i.e., MPV positions) (double),
%           integration time (double)
%
% Output:   "bindata" structure containing binned (i.e., integrated) 
%           data of all gas types and variables, 
%           overview figure
%
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

function [data, data_ch4, bindata] = bin_data(data,data_ch4,gasnames,gastypeflag,binning_interval)

tic;fprintf('Running function bin_data...\n');

% find start time and end time
bindata.start_time=max(data.start_time,data_ch4.start_time); 
bindata.end_time=min(data.end_time,data_ch4.end_time);  

% define number of bins
bindata.duration=bindata.end_time-bindata.start_time; %hours
bindata.nbin=round(bindata.duration./(binning_interval/3600));

bin=binning_interval/3600;      %convert binning interval to hours
nbin=bindata.nbin;              %no. bins
t_start=bindata.start_time;     %start time

%% check if CH4 data file was uploaded
if data_ch4.data_idx==0
    bindata.data_ch4_idx=0;
end

if data_ch4.data_idx==1
    bindata.data_ch4_idx=1;
end

%% average data N2O

for j=1:nbin   
    for z=1:height(data.plateau)
        if data.plateau(z,1)>t_start+bin*(j-1) && data.plateau(z,1)<t_start+bin*j
                data.plateau_idx(z,1)=j;
        end
    end
end

 for k=1:width(data.plateau)
    for j=1:nbin   
        test=find(data.plateau_idx == j);
        bindata.n2o(j,k)=mean(data.plateau(test,k),'omitnan');
    end
 end
   
%% average data CH4

for j=1:nbin   
    for z=1:height(data_ch4.all)
        if data_ch4.all(z,1)>t_start+bin*(j-1) && data_ch4.all(z,1)<t_start+bin*j
                data_ch4.all_idx(z,1)=j;
        end
    end
end
    
 for k=1:width(data_ch4.all)
    for j=1:nbin   
        test=find(data_ch4.all_idx == j);
        bindata.ch4(j,k)=mean(data_ch4.all(test,k),'omitnan');
    end
 end

%% select plateau binned CH4 data

for i=1:height(bindata.n2o)
    if isnan(bindata.n2o(i,:))
        bindata.ch4(i,:)=NaN;
    end
end

%% merge binned data vectors

bindata.all=[(bindata.n2o(:,1)-bindata.start_time) bindata.n2o(:,3:end) bindata.ch4(:,3:6)];

bindata.varlabels={'Time [h]', data.varlabels{1,3:end}, data_ch4.varlabels{1,3:6}};

%% define gas types
  
  for k=1:width(gasnames)
        bindata.(gasnames{1,k})=NaN(height(bindata.all),width(bindata.all));
        for i=1:height(bindata.all)
             if bindata.all(i,11)==gastypeflag(1,k)
                  bindata.(gasnames{1,k})(i,:)=bindata.all(i,:); 
             end
        end
  end
  
%% Remove empty rows 

  for k=1:width(gasnames)
      bindata.(gasnames{1,k})=rmmissing(bindata.(gasnames{1,k}),1,'MinNumMissing',width(bindata.all));
  end

%%
time_elapsed=toc; fprintf('bin_data completed (execution time: %1.2f s)\n',time_elapsed); 

end